﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Diagnostics;
using System.Linq;

using Microsoft.Extensions.Configuration;

using Platform.Logging;
using Platform.Sql.ChangeMonitor;

namespace Platform.Configuration.DatabaseConfiguration
{
    public class DatabaseConfigurationProvider : ConfigurationProvider, IDisposable
    {
        private bool _IsEnabled;
        private string _ConnectionString;
        private string _SqlQueryString;
        private IConfiguration _Configuration;
        private QueryChangeMonitor _Monitor;

        public DatabaseConfigurationProvider( IConfiguration configuration )
        {
            if (!bool.TryParse(configuration.GetValue("Configuration:Database:IsEnabled", "true"), out _IsEnabled)) _IsEnabled = true;
            _ConnectionString = configuration.GetValue("Configuration:Database:ConnectionString", "Data Source=.;Initial Catalog=Configuration;Integrated Security=true");
            _SqlQueryString = configuration.GetValue("Configuration:Database:Query", "SELECT Name, Value FROM ConfigurationSettings");
            _Configuration = configuration;
            StartListening();
        }

        ~DatabaseConfigurationProvider()
        {
            Dispose(false);
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (disposing)
            {
                StopListening();
            }
        }

        private void StartListening()
        {
            try
            {
                if (_Monitor != null) return;
                _Monitor = new QueryChangeMonitor(null);
                _Monitor.Name = "Database Configuration";
                _Monitor.Changed += OnMonitorChanged;
                _Monitor.Start(_ConnectionString, _SqlQueryString);
            }
            catch( Exception ex )
            {
                SimpleFileLogger.WriteLine(_Configuration.GetSection("Logging"), "Monitor of Configuration Database Failed: {0}\r\n{1}", ex.Message, ex.ToString());
                Debug.WriteLine("Monitor of Configuration Database Failed: {0}\r\n{1}", ex.Message, ex.ToString());
                _Monitor.Dispose();
                _Monitor = null;
            }
        }

        private void StopListening()
        {
            try
            {
                if (_Monitor == null) return;
                _Monitor.Changed -= OnMonitorChanged;
                _Monitor.Stop();
                _Monitor.Dispose();
                _Monitor = null;
            }
            catch ( Exception ex )
            {
                SimpleFileLogger.WriteLine(_Configuration.GetSection("Logging"), "Monitor dispose of Configuration Database Failed: {0}\r\n{1}", ex.Message, ex.ToString());
                Debug.WriteLine("Monitor dispose of Configuration Database Failed: {0}\r\n{1}", ex.Message, ex.ToString());
                _Monitor = null;
            }
        }

        private void OnMonitorChanged(object sender, EventArgs e)
        {
            try
            {
                Load();
                OnReload();
            }
            catch( Exception ex )
            {
                SimpleFileLogger.WriteLine(_Configuration.GetSection("Logging"), "Reload of Configuration Database Failed: {0}\r\n{1}", ex.Message, ex.ToString());
                Debug.WriteLine("Reload of Configuration Database Failed: {0}\r\n{1}", ex.Message, ex.ToString());
            }
        }

        /// <summary>
        /// Load settings from provider
        /// </summary>
        public override void Load()
        {
            // configuration values store
            var dictionary = new Dictionary<string, string>();

            try
            {
                using (var connection = new SqlConnection(_ConnectionString))
                using (var command = connection.CreateCommand())
                {
                    // open connection
                    if (connection.State == System.Data.ConnectionState.Closed)
                        connection.Open();

                    // create command
                    command.CommandText = _SqlQueryString;
                    command.CommandTimeout = 20;
                    command.CommandType = CommandType.Text;

                    // execute command
                    using (var reader = command.ExecuteReader())
                    {
                        // keep reading
                        while( reader.Read() )
                        {
                            var id = reader.GetString(0);
                            var value = reader.GetString(1);
                            dictionary[id] = value;
                        }
                    }
                }

                // store values
                Data = dictionary;
            }
            catch( Exception ex )
            {
                SimpleFileLogger.WriteLine(_Configuration.GetSection("Logging"), "Load of Configuration Database Failed: {0}\r\n{1}", ex.Message, ex.ToString());
                Debug.WriteLine("Load of Configuration Database Failed: {0}\r\n{1}", ex.Message, ex.ToString());
            }
        }
    }    
}
