﻿using System.Text;

using Platform.Net;
using Platform.Extensions;

namespace Platform.Uri
{
    /// <summary>
    /// Construct new Uri
    /// </summary>
    public partial class NetUri : NetValue
    {
        #region Construction & Fields

        private string _Scheme;
        private string _Fragment;

        private NetUriAuthority _Authority;
        private NetUriPath _Path;
        private NetUriQuery _Query;

        private const string _SchemePattern = @"[a-zA-Z][-+.a-zA-Z0-9]";
        private const string _FragmentSet = @"-0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ._~:?#@!$&',;=/[]<>*+";
        private const string _SchemeSet = @"0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ-+.";

        /// <summary>
        /// Construct new Uri
        /// </summary>
        public NetUri()
        {
        }

        /// <summary>
        /// Construct new uri
        /// </summary>
        /// <param name="uri"></param>
        public NetUri(string source)
            : base( source )
        {
        }

        /// <summary>
        /// Construct new embedded uri
        /// </summary>
        /// <param name="reader"></param>
        public NetUri( NetStringReader reader )
            : base( reader )
        {
        }

        /// <summary>
        /// Construct from uri and path
        /// </summary>
        /// <param name="uri"></param>
        /// <param name="path"></param>
        public NetUri(string uri, string path)
            : this( uri )
        {
            AddRelativePath(path);
        }

        /// <summary>
        /// Initialize part
        /// </summary>
        protected override void Initialize()
        {
            _Scheme = "";
            _Authority = new NetUriAuthority();
            _Path = new NetUriPath();
            _Query = new NetUriQuery();
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets or sets the decoded scheme
        /// </summary>
        public string Scheme
        {
            get { Decode(); return _Scheme; }
            set { Decode(); _Scheme = value; }
        }

        /// <summary>
        /// Gets or sets the encoded authority (may start with //)
        /// </summary>
        public NetUriAuthority Authority
        {
            get { Decode(); return _Authority; }
            set { Decode(); _Authority = value; }
        }

        /// <summary>
        /// Gets or sets the encoded Path (may start with /)
        /// </summary>
        public NetUriPath Path
        {
            get { Decode(); return _Path; }
            set { Decode(); _Path = value; }
        }

        /// <summary>
        /// Gets or sets the encoded Query (Should start with ?)
        /// </summary>
        public NetUriQuery Query
        {
            get { Decode(); return _Query; }
            set { Decode(); _Query = value; }
        }

        /// <summary>
        /// Gets or sets the decoded fragment
        /// </summary>
        public string Fragment
        {
            get { Decode(); return _Fragment; }
            set { Decode(); _Fragment = value; }
        }

        /// <summary>
        /// Gets the encoded scheme string
        /// </summary>
        public string SchemeString
        {
            get { return string.IsNullOrEmpty(_Scheme) ? "" : _Scheme + ":"; }
        }


        /// <summary>
        /// Gets the encoded fragment string
        /// </summary>
        public string FragmentString
        {
            get { return string.IsNullOrEmpty(_Fragment) ? "" : "#" + _Fragment.UriEncode(_FragmentSet, false); }
        }

        /// <summary>
        /// Gets the relative string (No scheme or authority)
        /// </summary>
        public string RelativeString
        {
            get { return Path.ToString() + Query.ToString() + FragmentString; }
        }

        #endregion

        #region String Casts

        /// <summary>
        /// Convert from string
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        public static implicit operator NetUri(string source)
        {
            return new NetUri( source );
        }

        /// <summary>
        /// Convert to string
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        public static implicit operator string(NetUri source)
        {
            if (source == null) return null;
            else return source.ToString();
        }

        #endregion

        #region Uri Casts

        /// <summary>
        /// Convert from Uri
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        public static implicit operator NetUri(System.Uri source)
        {
            return new NetUri(source.ToString());
        }

        /// <summary>
        /// Convert to Uri
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        public static implicit operator System.Uri(NetUri source)
        {
            if (source == null) return null;
            else return new System.Uri( source.ToString() );
        }

        #endregion

        #region Encoding / Decoding

        /// <summary>
        /// Decode uri into parts
        /// </summary>
        protected override void OnDecode( NetStringReader reader )
        {
            // get scheme
            _Scheme = reader.ReadWhile( _SchemeSet.ToCharArray() );
            reader.Read(":");
                            
            // get authority
            _Authority = new NetUriAuthority(reader.Read("//") + reader.ReadAllOrUntil('/', '?', '#'));

            // get path
            if (reader.StartsWith("/")==-1) _Path = new NetUriPath();
            else _Path = new NetUriPath(reader.ReadAllOrUntil('?', '#'));

            // get query
            if (reader.StartsWith("?")==-1) _Query = new NetUriQuery();
            else _Query = new NetUriQuery(reader.ReadAllOrUntil('#'));

            // get fragment
            if (reader.ReadPattern("#") != 0) _Fragment = "";
            else _Fragment = reader.ReadAll().UriDecode(false);
        }

        /// <summary>
        /// Convert to Uri string
        /// </summary>
        /// <returns></returns>
        protected override void OnEncode( NetStringWriter writer )
        {
            // add all parts
            writer.Write(SchemeString);
            var authority = Authority.Encode();
            writer.Write(authority);
            var path = Path.Encode();

            // path should be prefixed if authority is given
            if ( authority != "" && path != "" && !path.StartsWith("/") )
                writer.Write("/");

            // write fragment
            writer.Write(path);
            writer.Write(Query);
            writer.Write(FragmentString);
        }

        #endregion

        #region Uri Tools

        /// <summary>
        /// Gets the root without the path
        /// </summary>
        public string Root
        {
            get
            {
                StringBuilder result = new StringBuilder();
                result.Append(Scheme);
                result.Append(":");
                result.Append(Authority);
                return result.ToString();
            }
        }


        /// <summary>
        /// Gets the resource only part
        /// </summary>
        public string Resource
        {
            get
            {
                StringBuilder result = new StringBuilder();
                result.Append(Path.ToString());
                result.Append(Query.ToString());
                result.Append(FragmentString);
                return result.ToString();
            }
        }

        /// <summary>
        /// Check if uri is child of other uri
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        public bool IsChildOf(NetUri uri)
        {
            if (!Scheme.IsEqualTo(uri.Scheme, true)) return false;
            if (!Authority.ToString().IsEqualTo(uri.Authority.ToString(), true)) return false;
            return Path.ToString().StartsWith(uri.Path.ToString());
        }

        /// <summary>
        /// Gets the relative path
        /// </summary>
        /// <param name="uri"></param>
        /// <returns></returns>
        public string GetRelativePath(NetUri uri)
        {
            return uri.Path.ToString().Trim('/').Substring(Path.ToString().Trim('/').Length);
        }

        /// <summary>
        /// Add relative path
        /// </summary>
        /// <param name="path"></param>
        public void AddRelativePath( string path )
        {
            if (string.IsNullOrEmpty(path)) return;
            if (Path == null) Path = new NetUriPath(path);
            else Path.AddRelativePath(path);
        }

        #endregion
    }
}
