﻿
using Platform.Net;
using Platform.Extensions;

namespace Platform.Uri
{
    /// <summary>
    /// Authority part of Uri
    /// </summary>
    public class NetUriAuthority : NetValue
    {
        #region Construction & Fields

        private bool _IsRoot;
        private string _Username;
        private string _Password;
        private NetUriHost _Host;
        private NetUriAuthorityParameters _Parameters;

        private const string _UsernameSet = @"-0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ._~!$&',;=()*+";
        private const string _PasswordSet = @"-0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ._~!$&',;=:()*+";

        /// <summary>
        /// Construct new empty root authority
        /// </summary>
        public NetUriAuthority()
        {
        }

        /// <summary>
        /// Construct new root authority
        /// </summary>
        /// <param name="source"></param>
        internal NetUriAuthority( string source )
            : base( source )
        {
        }

        /// <summary>
        /// Initialize
        /// </summary>
        protected override void Initialize()
        {
            _Host = new NetUriHost();
            _Parameters = new NetUriAuthorityParameters();
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets or sets if this is a root authority
        /// </summary>
        public bool IsRoot
        {
            get { Decode(); return _IsRoot; }
            set { Decode(); _IsRoot = value; }
        }

        /// <summary>
        /// Gets or sets the user info
        /// </summary>
        public string Username 
        {
            get { Decode(); return _Username; }
            set { Decode(); _Username = value; }
        }

        /// <summary>
        /// Gets or sets the password
        /// </summary>
        public string Password 
        {
            get { Decode(); return _Password; }
            set { Decode(); _Password = value; }
        }

        /// <summary>
        /// Gets or sets the host
        /// </summary>
        public NetUriHost Host 
        {
            get { Decode(); return _Host; }
            set { Decode(); _Host = value; }
        }

        /// <summary>
        /// Gets or sets the host
        /// </summary>
        public NetUriAuthorityParameters Parameters
        {
            get { Decode(); return _Parameters; }
            set { Decode(); _Parameters = value; }
        }

        /// <summary>
        /// Gets the host string
        /// </summary>
        public string HostString
        {
            get { return Host == null ? "" : Host.ToString(); }
        }

        /// <summary>
        /// Gets the parameters string
        /// </summary>
        public string ParametersString
        {
            get { return Parameters == null ? "" : Parameters.ToString(); }
        }

        #endregion

        #region String Casts

        /// <summary>
        /// Convert from string
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        public static implicit operator NetUriAuthority(string source)
        {
            return new NetUriAuthority(source);
        }

        /// <summary>
        /// Convert to string
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        public static implicit operator string(NetUriAuthority source)
        {
            if (source == null) return null;
            else return source.ToString();
        }

        #endregion

        #region Encoding / Decoding

        /// <summary>
        /// Decode uri into parts
        /// </summary>
        protected override void OnDecode(NetStringReader reader)
        {
            var prefix = reader.ReadPattern( "//" );

            // set root flag
            _IsRoot = prefix == 0;

            // check for credentails
            var credentials = reader.ReadIfTerminated('@');
            reader.Read("@");

            // find end of username:password
            if (credentials != "")
            {
                // split into username and password
                int seperator = credentials.IndexOf(':');
                if (seperator == -1)
                {
                    _Username = credentials.UriDecode(false);
                    _Password = null;
                }
                else
                {
                    _Username = credentials.Substring(0, seperator).UriDecode(false);
                    _Password = credentials.Substring(seperator + 1).UriDecode(false);
                }
            }

            // get host
            _Host = reader.ReadAllOrUntil(';');
            reader.Read("@");

            // get parameters
            _Parameters = new NetUriAuthorityParameters( reader.ReadAll() );
        }

        /// <summary>
        /// Convert to Uri string
        /// </summary>
        /// <returns></returns>
        protected override void OnEncode(NetStringWriter writer)
        {
            // add root specifier
            if (_IsRoot) writer.WriteFormat("//");

            // add username
            if (_Username != null)
                writer.Write(_Username.UriEncode(_UsernameSet, false));

            // add password
            if (_Password != null)
            {
                writer.Write(":");
                writer.Write(_Password.UriEncode(_PasswordSet, false));
            }

            // add seperator
            if (_Username != null || _Password != null)
                writer.Write("@");

            // add host
            writer.Write(HostString);

            // add parameters
            writer.Write(ParametersString);
        }

        #endregion
    }
}
