﻿using Platform.Net;
using Platform.Extensions;
using System.Net.Sockets;

namespace Platform.Uri
{
    /// <summary>
    /// Construct new NetUriHost
    /// </summary>
    public class NetUriHost : NetValue
    {
        #region Construction & Fields

        private bool _IsAddress;
        private string _Address;
        private NetUriHostName _Name;
        private int? _Port;

        public NetUriHost()
        {
        }

        internal NetUriHost(string source)
            : base(source)
        {
        }

        public NetUriHost(string name, int? port)
        {
            _IsAddress = IsAddressString(name);
            _Name = _IsAddress ? null : name;
            _Address = _IsAddress ? name : null;
            _Port = port;
        }

        protected override void Initialize()
        {
            _Name = new NetUriHostName();
        }

        #endregion

        #region Properties

        /// <summary>
        /// Returns true if address
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        private static bool IsAddressString(string value)
        {
            if (string.IsNullOrEmpty(value)) return false;
            return value.Contains(":") || char.IsDigit(value[0]);
        }

        /// <summary>
        /// Gets if this represents an address
        /// </summary>
        public bool IsAddress
        {
            get { Decode(); return _IsAddress && _Address != null; }
        }

        /// <summary>
        /// Gets if this represents a name
        /// </summary>
        public bool IsName
        {
            get { Decode(); return !_IsAddress && _Name != null; }
        }

        /// <summary>
        /// Gets the address
        /// </summary>
        public string Address
        {
            get { Decode(); return _Address; }
            set { Decode(); _Address = value; _Name = null; _IsAddress = true; }
        }

        /// <summary>
        /// Get address family
        /// </summary>
        public AddressFamily? AddressFamily
        {
            get { 
                Decode();
                if (!_IsAddress || string.IsNullOrEmpty(_Address)) return null;
                if (_Address.Contains(":")) return System.Net.Sockets.AddressFamily.InterNetworkV6;
                return System.Net.Sockets.AddressFamily.InterNetwork;
            }
        }

        /// <summary>
        /// Gets the name
        /// </summary>
        public NetUriHostName Name
        {
            get { Decode(); return _Name; }
            set { Decode(); _Name = value; _Address = null; _IsAddress = false; }
        }

        /// <summary>
        /// Gets the optional port number
        /// </summary>
        public int? Port
        {
            get { Decode(); return _Port; }
            set { Decode(); _Port = value; }
        }

        /// <summary>
        /// Gets the full name
        /// </summary>
        public string NameOrAddress
        {
            get { Decode(); return IsAddress ? _Address.ToString() : _Name.ToString(); }
            set
            {
                // no hostname or address
                if( value == null || value == "" )
                {
                    _Address = null;
                    _Name = null;
                    _IsAddress = false;
                    return;
                }

                // check for IP4 or IP6 address
                if ( IsAddressString(value)  )
                {
                    _Address = value;
                    _Name = null;
                    _IsAddress = true;
                    return;
                }

                // normal address
                _Name = value;
                _Address = null;
                _IsAddress = false;
            }
        }

        /// <summary>
        /// Gets a comparable full string
        /// </summary>
        public string GetHostString( int port )
        {
            return string.Format("{0}:{1}", NameOrAddress, Port ?? port).ToLower();
        }
        
        #endregion

        #region String Casts

        /// <summary>
        /// Convert from string
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        public static implicit operator NetUriHost(string source)
        {
            return new NetUriHost(source);
        }

        /// <summary>
        /// Convert to string
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        public static implicit operator string(NetUriHost source)
        {
            if (source == null) return null;
            else return source.ToString();
        }

        #endregion

        #region Encoding / Decoding

        /// <summary>
        /// Decode uri into parts
        /// </summary>
        protected override void OnDecode(NetStringReader reader)
        {
            // check for IPv6
            if( reader.ReadPattern("[") == 0 )
            {
                _IsAddress = true;
                _Address = reader.ReadAllOrUntil(']');
                _Name = null;
                reader.Read("]");
            }
            else
            {
                var address = reader.ReadAllOrUntil(':');
                if (!string.IsNullOrEmpty(address) && char.IsDigit(address[0]))
                {
                    _Address = address;
                    _Name = null;
                    _IsAddress = true;
                }
                else
                {
                    _Address = null;
                    _Name = address;
                    _IsAddress = false;
                }
            }

            // read port
            if (reader.ReadPattern(":") != 0) _Port = null;
            else _Port = reader.ReadAll().ToInt32(10, null);
        }

        /// <summary>
        /// Convert to Uri string
        /// </summary>
        /// <returns></returns>
        protected override void OnEncode(NetStringWriter writer)
        {
            // add all parts
            if( !_IsAddress ) writer.Write(_Name.ToString());
            else if( _Address.Contains(":") ) writer.Write("[" + _Address.ToString() + "]" );
            else writer.Write(_Address.ToString());

            // add port
            if (_Port != null) writer.WriteFormat(":{0}", _Port.Value);
        }

        #endregion

        #region Match

        public override bool IsMatch(NetValue value)
        {
            if (value == null || value.GetType() != GetType()) return false;
            var other = value as NetUriHost;

            if (!NameOrAddress.IsEqualTo(other.NameOrAddress, true)) return false;
            if (Port != other.Port) return false;

            return true;
        }

        #endregion
    }
}
