﻿using System;
using System.Collections.Generic;
using System.Linq;

using Platform.Net;
using Platform.Extensions;

namespace Platform.Uri
{
    public class NetUriHostName : NetValue, IEnumerable<string>
    {
        #region Construction and Fields

        private string _Device;
        private string[] _Domain;
        private string _TopLevel;

        private const string _DeviceSet = @"-0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_~!$&',=()*";
        private const string _DomainSet = @"-0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_~!$&',=()*";
        private const string _TopLevelSet = @"-0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_~!$&',=()*";

        public NetUriHostName()
        {
        }

        public NetUriHostName(string source)
            : base(source)
        {
        }

        protected override void Initialize()
        {
            _Domain = new string[0];
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets or sets the device name part
        /// </summary>
        public string Device
        {
            get { Decode(); return _Device; }
            set { Decode(); _Device = value; }
        }

        /// <summary>
        /// Gets or sets the domain name part
        /// </summary>
        public string[] Domain
        {
            get { Decode(); return _Domain; }
            set { Decode(); _Domain = value; }
        }

        /// <summary>
        /// Gets or sets the top level part
        /// </summary>
        public string TopLevel
        {
            get { Decode(); return _TopLevel; }
            set { Decode(); _TopLevel = value; }
        }

        /// <summary>
        /// Gets the fullname
        /// </summary>
        public string Fullname
        {
            get
            {
                // decode
                Decode();

                // get source parts
                List<string> parts = new List<string>();
                if (!string.IsNullOrEmpty(_Device)) parts.Add(_Device);
                if (_Domain != null) parts.AddRange(_Domain);
                if (!string.IsNullOrEmpty(_TopLevel)) parts.Add(_TopLevel);

                // return result
                return string.Join(".", parts.ToArray());
            }
        }

        /// <summary>
        /// Get name by index
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public string this[ int index ]
        {
            get { return this.Skip(index).First(); }
            set
            {
                // check within range
                if( index >= Count || index < 0 ) 
                    throw new IndexOutOfRangeException();

                // set proper item
                if (index == 0) Device = value;
                else if (index == Count - 1) TopLevel = value;
                else if( Domain != null ) Domain[index - 1] = value;
            }
        }

        /// <summary>
        /// Get number if elements
        /// </summary>
        public int Count
        {
            get 
            {
                Decode();
                return this.Count(); 
            }
        }

        #endregion

        #region String Casts

        /// <summary>
        /// Convert from string
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        public static implicit operator NetUriHostName(string source)
        {
            return new NetUriHostName(source);
        }

        /// <summary>
        /// Convert to string
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        public static implicit operator string(NetUriHostName source)
        {
            if (source == null) return null;
            else return source.ToString();
        }

        #endregion

        #region Encoding / Decoding

        /// <summary>
        /// Decode uri into parts
        /// </summary>
        protected override void OnDecode(NetStringReader reader)
        {
            // get parts
            var parts = new List<string>();
            while( !reader.EndOfString )
            {
                var value = reader.ReadWhile(_DeviceSet.ToCharArray() ).UriDecode(false);
                if( value != "" ) parts.Add( value );
                if (reader.ReadPattern(".") != 0) break;
            }

            // get device name
            if (parts.Count == 0) _Device = null;
            else if (parts.Count == 1) _Device = parts[0];
            else if (parts.Count == 2) _Device = null;
            else _Device = parts[0];

            // get domain names
            if (parts.Count == 0) _Domain = new string[0];
            else if (parts.Count == 1) _Domain = new string[0];
            else if (parts.Count == 2) _Domain = new string[] { parts[0] };
            else _Domain = parts.Skip(1).Take(parts.Count - 2).ToArray();

            // get top level
            if (parts.Count == 0) _TopLevel = null;
            else if (parts.Count == 1) _TopLevel = null;
            else _TopLevel = parts.Last();
        }

        /// <summary>
        /// Convert to Uri string
        /// </summary>
        /// <returns></returns>
        protected override void OnEncode(NetStringWriter writer)
        {
            // get source parts
            List<string> parts = new List<string>();
            if (!string.IsNullOrEmpty(_Device)) parts.Add(_Device.UriEncode(_DeviceSet, false));
            if (_Domain != null) parts.AddRange(_Domain.Select(d => d.UriEncode(_DomainSet, false)));
            if (!string.IsNullOrEmpty(_TopLevel)) parts.Add(_TopLevel.UriEncode(_TopLevelSet, false));

            // return result
            writer.Write(string.Join(".", parts.ToArray()));
        }

        #endregion

        #region IEnumerable

        public IEnumerator<string> GetEnumerator()
        {
            // decode first
            Decode();

            // return all items
            if( _Device != null ) yield return _Device;
            if( Domain != null ) foreach( var domain in _Domain ) yield return domain;
            if( _TopLevel != null ) yield return _TopLevel;
        }

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }

        #endregion
    }
}
