﻿
using Platform.Net;

namespace Platform.Uri
{
    public class NetUriQuery : NetUriParameters<NetUriQuerySegment>
    {
        #region Construction & Fields

        public NetUriQuery()
        {
        }

        internal protected NetUriQuery(string source)
            : base( source )
        {
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets the seperation type
        /// </summary>
        protected override NetListSeperationType Seperation { get { return NetListSeperationType.Seperated; } }

        /// <summary>
        /// Gets the seperator
        /// </summary>
        protected override char Seperator { get { return '&'; } }

        #endregion

        #region String Casts

        /// <summary>
        /// Convert from string
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        public static implicit operator NetUriQuery(string source)
        {
            return new NetUriQuery(source);
        }

        /// <summary>
        /// Convert to string
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        public static implicit operator string(NetUriQuery source)
        {
            if (source == null) return null;
            else return source.ToString();
        }

        #endregion

        #region Encode / Decode

        /// <summary>
        /// Create list item
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        protected override NetUriQuerySegment CreateListItem(NetStringReader reader)
        {
            return new NetUriQuerySegment(reader.ReadAll());
        }

        /// <summary>
        /// Create parameter
        /// </summary>
        /// <param name="name"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        protected override NetUriQuerySegment CreateParameter(string name, string value)
        {
            return new NetUriQuerySegment(name, value);
        }

        /// <summary>
        /// Decode no prefix
        /// </summary>
        protected override void OnDecode(NetStringReader reader)
        {
            bool question = reader.Read('?');
            base.OnDecode(reader);
        }

        /// <summary>
        /// Encode no prefix
        /// </summary>
        /// <returns></returns>
        protected override void OnEncode(NetStringWriter writer)
        {
            if (Count > 0) writer.Write('?');
            base.OnEncode(writer);
        }

        #endregion

        #region Tools

        /// <summary>
        /// Add query element
        /// </summary>
        /// <param name="name"></param>
        /// <param name="value"></param>
        public void Add( string name, string value )
        {
            Add(new NetUriQuerySegment(name, value));
        }

        /// <summary>
        /// Merge query arguments
        /// </summary>
        /// <param name="items"></param>
        public void Merge(string items)
        {
            AddRange( new NetUriQuery(items) );
        }

        /// <summary>
        /// Gets a segment parameter
        /// </summary>
        /// <param name="segment"></param>
        /// <param name="parameter"></param>
        /// <returns></returns>
        public string GetSegmentParameter(string segment, string parameter)
        {
            var item = GetParameter(segment);
            if (item == null) return null;
            return item.Parameters[parameter];
        }

        /// <summary>
        /// Gets a segment parameter
        /// </summary>
        /// <param name="segment"></param>
        /// <param name="parameter"></param>
        /// <returns></returns>
        public string GetSegmentParameter(string segment, string parameter, string @default)
        {
            var item = GetParameter(segment);
            if (item == null) return @default;
            return item.Parameters[parameter, @default];
        }

        /// <summary>
        /// Sets a segment parameter
        /// </summary>
        /// <param name="segment"></param>
        /// <param name="parameter"></param>
        /// <param name="value"></param>
        public void SetSegmentParameter(string segment, string parameter, string value)
        {
            var item = GetParameter(segment);
            if (item == null) Add(item = new NetUriQuerySegment(segment, null));
            if (value != null) item.Parameters[parameter] = value;
            else item.Parameters.Remove(parameter);
        }

        #endregion
    }
}
